import React from "react";
import { BrowserRouter as Router, Routes, Route, Navigate, Outlet } from "react-router-dom";
import { AuthProvider, useAuth } from "./context/AuthContext";
import Login from "./components/Login";
import Register from "./components/Register";
import PendingApproval from "./components/PendingApproval";
import AdminPanel from "./components/AdminPanel";
import Dashboard from "./Dashboard";

// Protected Route Wrapper
const ProtectedRoute = ({ children, requireAdmin = false }) => {
    const { user, userData, loading } = useAuth();

    if (loading) return <div className="p-8">Loading...</div>;

    if (!user) {
        return <Navigate to="/login" replace />;
    }

    // Identify if user data is loaded (if logged in, we should have userData or be fetching it)
    // If userData is null but user exists, it might be fetching. 
    // But AuthContext sets loading=false only after auth check. 
    // AuthContext should fetch userData before setting loading=false if user exists.

    if (!userData) {
        // Fallback if userData fetch failed or not ready
        return <div className="p-8">Loading User Data...</div>;
    }

    if (!userData.approved) {
        return <Navigate to="/pending" replace />;
    }

    if (requireAdmin && userData.role !== 'admin') {
        return <Navigate to="/" replace />;
    }

    return children ? children : <Outlet />;
};

const PublicOnlyRoute = ({ children }) => {
    const { user, loading } = useAuth();
    if (loading) return <div>Loading...</div>;
    if (user) return <Navigate to="/" replace />;
    return children;
};

const PendingRoute = ({ children }) => {
    const { user, userData, loading } = useAuth();
    if (loading) return <div>Loading...</div>;
    if (!user) return <Navigate to="/login" replace />;
    if (userData?.approved) return <Navigate to="/" replace />;
    return children;
};

export default function App() {
    return (
        <AuthProvider>
            <Router>
                <Routes>
                    <Route path="/login" element={<PublicOnlyRoute><Login /></PublicOnlyRoute>} />
                    <Route path="/register" element={<PublicOnlyRoute><Register /></PublicOnlyRoute>} />

                    <Route path="/pending" element={<PendingRoute><PendingApproval /></PendingRoute>} />

                    {/* Protected Routes */}
                    <Route path="/" element={<ProtectedRoute><Dashboard /></ProtectedRoute>} />

                    <Route path="/admin" element={<ProtectedRoute requireAdmin={true}><AdminPanel /></ProtectedRoute>} />

                    <Route path="*" element={<Navigate to="/" />} />
                </Routes>
            </Router>
        </AuthProvider>
    );
}
